"use client";

import React from "react";
import Image from "next/image";
import { Shield, Lock, History, Award } from "lucide-react";

interface SecurityFeature {
  icon: React.ReactNode;
  title: string;
  description: string;
}

const SecuritySection: React.FC = () => {
  const securityFeatures: SecurityFeature[] = [
    {
      icon: <Lock className="w-6 h-6" />,
      title: "Bank-Grade Encryption",
      description: "Your files and data are protected with AES-256 encryption, ensuring privacy across all Clikkle apps.",
    },
    {
      icon: <Shield className="w-6 h-6" />,
      title: "Advanced Access Control",
      description: "Manage team permissions with precision. Control who views, edits, and shares sensitive business documents.",
    },
    {
      icon: <History className="w-6 h-6" />,
      title: "Automatic Data Backup",
      description: "Never lose a file again. Clikkle automatically backs up your work with easy one-click recovery options.",
    },
    {
      icon: <Award className="w-6 h-6" />,
      title: "Global Compliance",
      description: "Fully GDPR & SOC2 compliant, meeting international standards to keep your organization safe and legal.",
    },
  ];

  return (
    <section className="w-full flex justify-center items-center bg-white dark:bg-[#1a1a1a] transition-colors duration-300">
      <div className="relative my-16 mx-4 md:mx-6 w-full max-w-[1200px]">
        {/* Glassmorphic Container */}
        <div className="bg-white/30 dark:bg-[#141414] backdrop-blur-xl border border-gray-200 dark:border-[#ffffff1a] rounded-[1.5rem] md:rounded-[2.5rem] shadow-xl transition-all duration-500 p-6 md:p-12 relative overflow-hidden">
          {/* Decorative background elements */}

          <div className="relative z-10">
            {/* Header */}
            <div className="text-center mb-10 md:mb-14">
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/80 dark:bg-[#1a1a1a]/80 backdrop-blur-sm border border-gray-200 dark:border-[#ffffff1a] mb-6 shadow-sm">
                <Shield className="w-3.5 h-3.5 text-[#286efb]" />
                <span className="text-[10px] md:text-xs font-bold text-gray-900 dark:text-white tracking-widest uppercase">
                  Uncompromised Security
                </span>
              </div>

              <h2 className="text-[28px] md:text-[42px] lg:text-[48px] font-bold text-gray-900 dark:text-white mb-4 leading-tight">
                Your Business Data.<br className="hidden md:block" />
                <span className="text-[#286efb]">Securely Protected.</span>
              </h2>

              <p className="text-sm md:text-base text-gray-600 dark:text-gray-300 max-w-2xl mx-auto leading-relaxed font-medium">
                Clikkle prioritizes your data integrity. From secure file storage to encrypted communication, 
                our platform is built to safeguard your organization&apos;s most valuable assets.
              </p>
            </div>

            {/* Security Content Grid: Features + Image (Swapped Layout) */}
            <div className="flex flex-col-reverse lg:grid lg:grid-cols-12 gap-8 lg:gap-10 mb-10 lg:mb-14 items-stretch">
              
              {/* Left: Security Features (Now on left, spanning 7 columns) */}
              <div className="lg:col-span-7 grid grid-cols-1 sm:grid-cols-2 gap-4 md:gap-6">
                {securityFeatures.map((feature, index) => (
                  <div
                    key={index}
                    className="group relative bg-white/60 dark:bg-[#1a1a1a] backdrop-blur-sm border border-gray-200 dark:border-[#ffffff1a] rounded-xl p-5 md:p-6 hover:bg-white dark:hover:bg-[#252432] transition-all duration-300 hover:shadow-lg hover:shadow-[#286efb]/5 hover:border-[#286efb]/30 flex flex-col justify-between h-full"
                  >
                    <div className="flex flex-col items-start text-left gap-4">
                      {/* Icon */}
                      <div className="flex items-center justify-center w-10 h-10 md:w-12 md:h-12 rounded-xl bg-[#eef5ff] dark:bg-[#286efb]/10 text-[#286efb] group-hover:scale-110 transition-transform duration-300 shadow-inner">
                        {feature.icon}
                      </div>

                      <div className="flex-1">
                        {/* Title */}
                        <h3 className="text-base md:text-lg font-bold text-gray-900 dark:text-white mb-2 group-hover:text-[#286efb] transition-colors">
                          {feature.title}
                        </h3>

                        {/* Description */}
                        <p className="text-xs md:text-sm text-gray-600 dark:text-gray-400 leading-relaxed">
                          {feature.description}
                        </p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* Right: Security Image (Now on right, spanning 5 columns) */}
              <div className="lg:col-span-5 relative rounded-2xl md:rounded-[2rem] overflow-hidden min-h-[300px] sm:min-h-[400px] lg:h-auto border border-gray-200 dark:border-[#ffffff1a] shadow-lg">
                <Image
                  src="https://images.pexels.com/photos/5380792/pexels-photo-5380792.jpeg"
                  alt="Clikkle Security Architecture"
                  fill
                  sizes="100vw"
                  className="object-cover object-center hover:scale-105 transition-transform duration-700"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-[#1a1a1a]/80 via-transparent to-transparent dark:from-black/80" />
                
                {/* Overlay Text on Image */}
                <div className="absolute bottom-0 left-0 p-6 w-full">
                  <div className="bg-white/10 backdrop-blur-md border border-white/20 rounded-xl p-3 md:p-4">
                    <p className="text-white text-xs md:text-sm font-medium">
                      &quot;Security isn&apos;t just a feature, it&apos;s our foundation.&quot;
                    </p>
                  </div>
                </div>
              </div>

            </div>

            {/* Trust Badge Row */}
            <div className="pt-8 md:pt-12 border-t border-gray-200 dark:border-[#ffffff1a]">
              <div className="flex flex-wrap items-center justify-center gap-8 md:gap-16">
                <div className="flex items-center gap-3 opacity-80 hover:opacity-100 transition-opacity">
                  <div className="flex items-center justify-center w-10 h-10 rounded-full bg-white dark:bg-[#252432] shadow-sm">
                    <Award className="w-5 h-5 text-[#286efb]" />
                  </div>
                  <div className="text-left">
                    <div className="text-sm font-bold text-gray-900 dark:text-white">SOC 2 Type II</div>
                    <div className="text-[10px] text-gray-500 dark:text-gray-400 uppercase tracking-wide">Certified</div>
                  </div>
                </div>

                <div className="flex items-center gap-3 opacity-80 hover:opacity-100 transition-opacity">
                  <div className="flex items-center justify-center w-10 h-10 rounded-full bg-white dark:bg-[#252432] shadow-sm">
                    <Shield className="w-5 h-5 text-[#286efb]" />
                  </div>
                  <div className="text-left">
                    <div className="text-sm font-bold text-gray-900 dark:text-white">GDPR</div>
                    <div className="text-[10px] text-gray-500 dark:text-gray-400 uppercase tracking-wide">Compliant</div>
                  </div>
                </div>

                <div className="flex items-center gap-3 opacity-80 hover:opacity-100 transition-opacity">
                  <div className="flex items-center justify-center w-10 h-10 rounded-full bg-white dark:bg-[#252432] shadow-sm">
                    <Lock className="w-5 h-5 text-[#286efb]" />
                  </div>
                  <div className="text-left">
                    <div className="text-sm font-bold text-gray-900 dark:text-white">AES-256</div>
                    <div className="text-[10px] text-gray-500 dark:text-gray-400 uppercase tracking-wide">Encryption</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default SecuritySection;
