"use client";
import React, { useEffect, useState, useRef } from "react";
import { Button, Typography, Divider } from "@mui/material";
import Link from "next/link";
import Image from "next/image";
import { appList } from "../components/appList";
import FileTransferForm from "../components/FileTransferForm";

const sidebarRoutes = {
  "/": "Home",
  "/overview": "Overview",
  "/features": "Features",
  "#pricing": "Pricing",
  "/getStarted": "Get Started",
};

interface UploadedFile {
  id: string;
  name: string;
  size: number;
  type: string;
  file: File;
}

const TransferHome: React.FC = () => {
  const [isAnimating, setIsAnimating] = useState(false);
  const [isVibrating, setIsVibrating] = useState(false);
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [tooltipText, setTooltipText] = useState(
    "Long press to send your folders"
  );
  const [uploadedFiles, setUploadedFiles] = useState<UploadedFile[]>([]);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    document.title = "Clikkle Files | Transfer Files Instantly";
    const favicon = document.querySelector(
      "link[rel='icon']"
    ) as HTMLLinkElement;
    if (favicon) {
      favicon.href =
        "https://cdn.clikkle.com/images/files/logo/2025/clikkle%20files.png";
    }

    // eslint-disable-next-line react-hooks/set-state-in-effect
    setIsAnimating(true);

    const tooltipInterval = setInterval(() => {
      setTooltipText((prev) =>
        prev === "Long press to send your folders"
          ? "Click to send your files"
          : "Long press to send your folders"
      );
    }, 3000);

    return () => {
      clearInterval(tooltipInterval);
    };
  }, []);

  const handleFileSelect = () => {
    setIsVibrating(true);
    setTimeout(() => setIsVibrating(false), 500);
    fileInputRef.current?.click();
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(event.target.files || []);
    const newFiles: UploadedFile[] = files.map((file) => ({
      id: Date.now() + Math.random().toString(),
      name: file.name,
      size: file.size,
      type: file.type,
      file: file,
    }));

    setUploadedFiles((prev) => [...prev, ...newFiles]);
    setShowUploadModal(true);
  };

  return (
    <div
      className="min-h-screen w-full relative overflow-hidden"
      style={{ backgroundColor: "#000000" }}
    >
      <div className="absolute top-0 left-0 w-full z-20 p-6 flex justify-between items-center">
        <div className="flex items-center gap-3">
          <Image
            src="https://cdn.clikkle.com/images/files/logo/2023/files-text.png"
            alt="Clikkle Files"
            className="h-8"
            width={32}
            height={32}
          />
        </div>
        <div className="flex items-center gap-4">
          <Button
            variant="outlined"
            style={{
              borderColor: "rgba(255, 255, 255, 0)",
              color: "#ffffff",
              fontSize: "14px",
              fontWeight: "500",
              textTransform: "none",
              borderRadius: "6px",
              padding: "6px 12px",
              backgroundColor: "rgba(255, 255, 255, 0.1)",
              backdropFilter: "blur(10px)",
              borderWidth: "1px",
            }}
          >
            <Link
              href="https://worksuite.clikkle.com/signup?referralApp=Clikkle%20Website&&?redirectto=https%3A%2F%2Ffiles.clikkle.com%2F"
              style={{
                textDecoration: "none",
                color: "inherit",
                display: "flex",
                alignItems: "center",
                gap: "4px",
              }}
            >
              Get Files Pro
              <span style={{ fontSize: "14px" }}>🚀</span>
            </Link>
          </Button>

          <div
            onClick={() => setIsMenuOpen(!isMenuOpen)}
            style={{
              width: "32px",
              height: "32px",
              display: "flex",
              flexDirection: "column",
              justifyContent: "center",
              alignItems: "center",
              gap: "4px",
              cursor: "pointer",
              position: "relative",
              zIndex: 30,
            }}
          >
            <div
              style={{
                width: "18px",
                height: "2px",
                backgroundColor: "#ffffff",
                borderRadius: "1px",
                transition: "all 0.3s ease",
                transform: isMenuOpen
                  ? "rotate(45deg) translateY(6px)"
                  : "rotate(0)",
              }}
            ></div>
            <div
              style={{
                width: "18px",
                height: "2px",
                backgroundColor: "#ffffff",
                borderRadius: "1px",
                transition: "all 0.3s ease",
                opacity: isMenuOpen ? 0 : 1,
              }}
            ></div>
            <div
              style={{
                width: "18px",
                height: "2px",
                backgroundColor: "#ffffff",
                borderRadius: "1px",
                transition: "all 0.3s ease",
                transform: isMenuOpen
                  ? "rotate(-45deg) translateY(-6px)"
                  : "rotate(0)",
              }}
            ></div>
          </div>
        </div>
      </div>

      {isMenuOpen && (
        <div
          className="fixed inset-0 z-20"
          style={{ backgroundColor: "rgba(0, 0, 0, 0.5)" }}
          onClick={() => setIsMenuOpen(false)}
        />
      )}

      <div
        style={{
          position: "fixed",
          top: 0,
          right: isMenuOpen ? 0 : "-300px",
          width: "300px",
          height: "100vh",
          backgroundColor: "#2c2f33",
          zIndex: 25,
          transition: "right 0.3s ease",
          boxShadow: isMenuOpen ? "-2px 0 10px rgba(0, 0, 0, 0.3)" : "none",
        }}
      >
        <div
          style={{
            padding: "8px",
            height: "100%",
            overflowY: "auto",
            backgroundColor: "#1a1a1a",
          }}
        >
          <div
            style={{
              display: "flex",
              alignItems: "center",
              justifyContent: "space-between",
              padding: "8px",
            }}
          >
            <div style={{ display: "flex", alignItems: "center", gap: "8px" }}>
              <Image
                src="https://cdn.clikkle.com/images/worksuite/2025/clikkle%20worksuite.png"
                alt="Clikkle"
                style={{ width: "24px", height: "24px" }}
                width={24}
                height={24}
              />
              <span
                style={{
                  color: "#ffffff",
                  fontSize: "18px",
                  fontWeight: "500",
                }}
              >
                Clikkle Worksuite
              </span>
            </div>
            <div
              onClick={() => setIsMenuOpen(false)}
              style={{
                cursor: "pointer",
                color: "#ffffff",
                fontSize: "20px",
                padding: "4px",
              }}
            >
              ✕
            </div>
          </div>

          <Divider
            style={{
              backgroundColor: "rgba(255, 255, 255, 0.1)",
              margin: "8px 0",
            }}
          />

          <div
            style={{
              padding: "16px",
              display: "flex",
              flexDirection: "column",
              gap: "24px",
            }}
          >
            {Object.entries(sidebarRoutes).map(([route, label]) => (
              <Typography
                key={route}
                style={{
                  color: "#8a9ba8",
                  fontSize: "16px",
                  cursor: "pointer",
                }}
              >
                {label}
              </Typography>
            ))}
            <Typography style={{ color: "#8a9ba8", fontSize: "16px" }}>
              More Tools
            </Typography>
          </div>

          <div style={{ display: "flex", flexDirection: "column" }}>
            {appList.map((app) => (
              <Link
                key={app.logo}
                href={app.link}
                onClick={() => setIsMenuOpen(false)}
                style={{
                  display: "flex",
                  alignItems: "center",
                  gap: "8px",
                  padding: "8px",
                  textDecoration: "none",
                  color: "#8a9ba8",
                  fontSize: "16px",
                }}
              >
                <Image
                  src={app.logo}
                  alt={app.name}
                  style={{ width: "20px", height: "20px" }}
                  width={20}
                  height={20}
                />
                {app.name}
              </Link>
            ))}
          </div>

          <Divider
            style={{
              backgroundColor: "rgba(255, 255, 255, 0.1)",
              margin: "16px 0",
            }}
          />

          <div
            style={{
              padding: "16px",
              display: "flex",
              flexDirection: "column",
              gap: "16px",
            }}
          >
            <Button
              variant="contained"
              href="https://accounts.clikkle.com/number-of-employee"
              style={{
                backgroundColor: "#3b82f6",
                color: "#ffffff",
                fontSize: "16px",
                fontWeight: "500",
                textTransform: "none",
              }}
            >
              Try Clikkle
            </Button>
            <Button
              variant="outlined"
              href="https://accounts.clikkle.com/number-of-employee"
              style={{
                borderColor: "#3b82f6",
                color: "#3b82f6",
                fontSize: "16px",
                fontWeight: "500",
                textTransform: "none",
              }}
            >
              Go to Plus
            </Button>
            <Button
              href="https://worksuite.clikkle.com/login?referralApp=Clikkle%20Website&&"
              style={{
                color: "#3b82f6",
                fontSize: "16px",
                fontWeight: "500",
                textTransform: "none",
              }}
            >
              Sign In
            </Button>
          </div>
        </div>
      </div>

      <div className="relative z-10 flex flex-col items-center justify-center min-h-screen px-4">
        <div
          className={`mb-8 transition-all duration-500 ease-out ${
            isAnimating
              ? "opacity-100 translate-y-0"
              : "opacity-0 translate-y-4"
          }`}
          style={{
            background: "#4a5058",
            padding: "8px 14px",
            borderRadius: "6px",
            border: "1px solid rgba(255, 255, 255, 0.1)",
            position: "relative",
          }}
        >
          <div
            style={{
              position: "absolute",
              bottom: "-6px",
              left: "50%",
              transform: "translateX(-50%)",
              width: 0,
              height: 0,
              borderLeft: "6px solid transparent",
              borderRight: "6px solid transparent",
              borderTop: "6px solid #4a5058",
            }}
          />
          <p
            style={{
              margin: 0,
              fontSize: "13px",
              color: "#ffffff",
              textAlign: "center",
            }}
          >
            {tooltipText}
          </p>
        </div>

        <div
          className={`relative transition-all duration-1000 ease-out ${
            isAnimating ? "opacity-100 scale-100" : "opacity-0 scale-95"
          }`}
          style={{
            width: "240px",
            height: "240px",
            borderRadius: "50%",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            position: "relative",
            cursor: "pointer",
          }}
        >
          <div
            className="absolute inset-0 rounded-full"
            style={{
              border: "2px solid rgba(107, 114, 128, 0.3)",
              animation: "subtle-ripple 2s ease-out infinite",
            }}
          />
          <div
            className="absolute inset-0 rounded-full"
            style={{
              border: "2px solid rgba(107, 114, 128, 0.2)",
              animation: "subtle-ripple-delayed 2s ease-out infinite 0.5s",
            }}
          />
          <div
            className="absolute inset-0 rounded-full"
            style={{
              border: "50px solid rgba(174, 186, 210, 0.91)",
              animation: "subtle-ripple 2s ease-out infinite 1s",
            }}
          />
          <div
            onClick={handleFileSelect}
            style={{
              width: "180px",
              height: "180px",
              borderRadius: "50%",
              background:
                "linear-gradient(135deg,rgb(0, 0, 0) 0%,rgb(0, 0, 0) 100%)",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
              border: "2px solid rgba(255, 255, 255, 0.39)",
              transform: isVibrating ? "scale(0.95)" : "scale(1)",
              animation: isVibrating ? "vibrate 0.5s ease-in-out" : "none",
              boxShadow: "0 4px 20px rgba(0, 0, 0, 0.3)",
            }}
          >
            <Image
              src="https://cdn.clikkle.com/images/files/logo/2025/clikkle%20files.png"
              alt="Files Icon"
              style={{ width: "90px", height: "90px" }}
              width={90}
              height={90}
            />
          </div>
        </div>
      </div>

      <input
        type="file"
        ref={fileInputRef}
        onChange={handleFileChange}
        style={{ display: "none" }}
        multiple
      />

      {showUploadModal && (
        <div
          style={{
            position: "fixed",
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            backgroundColor: "rgba(0, 0, 0, 0.7)",
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            zIndex: 50,
            padding: "20px",
          }}
          onClick={() => setShowUploadModal(false)}
        >
          <div
            style={{
              backgroundColor: "#f8f9fa",
              borderRadius: "12px",
              width: "100%",
              maxWidth: "900px",
              maxHeight: "80vh",
              overflow: "auto",
              boxShadow: "0 25px 50px -12px rgba(0, 0, 0, 0.25)",
              position: "relative",
            }}
            onClick={(e) => e.stopPropagation()}
          >
            <button
              onClick={() => setShowUploadModal(false)}
              style={{
                position: "absolute",
                top: "20px",
                right: "20px",
                background: "transparent",
                border: "none",
                fontSize: "24px",
                cursor: "pointer",
                color: "#666",
                zIndex: 51,
                width: "32px",
                height: "32px",
                borderRadius: "50%",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
              }}
            >
              ×
            </button>
            <div style={{ padding: "20px" }}>
              <FileTransferForm initialFiles={uploadedFiles} />
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default TransferHome;
