"use client";
import React, { useEffect, useRef, useState } from "react";
import { Button } from "../components/button";
import Link from "next/link";
import Image from "next/image";

import { useTheme } from "../atoms/theme";
import { ArrowRightIcon, Ellipsis, Logs } from "lucide-react";
import { FaArrowRight } from "react-icons/fa";
import ThemeSwitch from "../components/themeSwitch";
import { apps } from "../constants/products";
import illustration from "../../public/images/banner.png";
import SidebarDrawer from "../components/sidebarDrawer";
import { footerColumns } from "../constants/footerText";
import AnnouncementBar from "../components/AnnouncementBar";
import { getCookie } from "../utils/cookies";
import ClikkleLogo from "@/components/ClikkleLogo";
import Footer from "@/components/Footer";
import { getEncodedReferrer } from "@/utils/utils";

const TermOfService: React.FC = () => {
  const { theme } = useTheme();

  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);

  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        productsRef.current &&
        !productsRef.current.contains(event.target as Node)
      ) {
        setProductsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);
  const [announcementOpen, setAnnouncementOpen] = useState(false);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (el && el.offsetHeight > 0) {
        setAnnouncementOpen(true);
      } else {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();

    const interval = setInterval(checkDismissedAnnouncements, 1000);

    return () => clearInterval(interval);
  }, []);
  return (
    <>
      <div className="min-h-screen font-sans bg-white dark:bg-[#1a1a1a]">
        <section className="w-full">
          <section className="relative w-full h-28 overflow-hidden">
            <div
              className={`fixed top-0 inset-x-0 z-50 ${
                scrolled
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : productsOpen || dotsOpen
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : ""
              }`}
            >
              <div
                ref={(el) => {
                  if (el) {
                    const height = el.getBoundingClientRect().height;

                    if (height !== 0) {
                      setAnnouncementOpen(true);
                    }
                  }
                }}
              >
                <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
              </div>
              <div className="container max-w-[1233px] mx-auto max-[1300px]:px-14 max-[1100px]:px-4">
                <nav
                  className={`relative flex items-center justify-between py-4`}
                >
                  <ClikkleLogo />

                  <div className="flex md:hidden items-center">
                    <div
                      className="flex items-center gap-0 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-[8px] shadow-sm"
                      onClick={toggleLeftDrawer}
                    >
                      <p className="text-sm font-medium text-gray-900 dark:text-white">
                        <Logs className="w-5 h-5 text-gray-900 dark:text-white mr-3" />
                      </p>

                      <p className="flex items-center  rounded-xl text-sm font-bold ">
                        Menu
                      </p>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-12 font-inter font-medium text-[18px]">
                    <Link
                      href="https://clikkle.com#features"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Features
                    </Link>
                    <Link
                      href="https://clikkle.com/#pricing"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Pricing
                    </Link>
                    <Link
                      href="/blogs"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Blog
                    </Link>
                    <div className="flex flex-row justify-center items-center">
                      <div className="relative" ref={productsRef}>
                        <button
                          onClick={() => setProductsOpen((o) => !o)}
                          className="cursor-pointer flex items-center gap-1 text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                        >
                          Products
                        </button>
                        {productsOpen && (
                          <div
                            className={`fixed ${
                              announcementOpen ? "top-[115px]" : "top-[75px]"
                            } 
              left-1/2 transform -translate-x-1/2 w-full bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
              shadow-lg p-6 border-b-2 border-gray-300 dark:border-[#575662] z-50 flex justify-center items-center pb-0`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                             <div className="flex flex-col gap-3">
                              {apps
                                .slice(0, Math.ceil(apps.length / 2))
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                            </div>

                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(
                                  Math.ceil(apps.length / 2),
                                  Math.ceil(apps.length / 2) + 2,
                                )
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                              <Link href="/products">
                                <Button
                                  style={{ marginTop: "5px" }}
                                  variant="ghost"
                                  className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                >
                                  View all products
                                  <ArrowRightIcon className="h-2 w-2 dark:text-white" />
                                </Button>
                              </Link>
                            </div>
                              <div className="hidden lg:flex flex-row items-center justify-center text-center gap-3">
                                <Image
                                  src={illustration}
                                  alt="Illustration"
                                  className="w-auto h-[175px]"
                                  width={400}
                                  height={175}
                                />
                                <div className="flex flex-col justify-center items-center h-full">
                                  <p className="text-sm text-gray-700 dark:text-gray-100 max-w-[200px] pb-5">
                                    Get started today the modern solution to
                                    power your business.
                                  </p>
                                  <Button
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                  >
                                    Get Started Today
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>

                      <div className="relative" ref={dotsRef}>
                        <Ellipsis
                          onClick={() => setDotsOpen((o) => !o)}
                          className="w-8 h-6 rounded-full p-[1px] cursor-pointer"
                        />
                        {dotsOpen && (
                          <div
                            className={`fixed ${
                              announcementOpen ? "top-[115px]" : "top-[75px]"
                            }  left-1/2 transform -translate-x-1/2 w-full 
    bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
    shadow-lg border-b-2 border-gray-300 dark:border-[#575662] 
    z-50 flex justify-center`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 py-6">
                              <div className="flex flex-col md:flex-row justify-between items-start gap-12">
                                <div className="space-y-12">
                                  <div className="max-w-[285px] space-y-2 mb-0">
                                    <h3 className="font-instrument-sans font-medium text-[18px] leading-[20px] text-gray-900 dark:text-[#eeeef0]">
                                      Get Started For Free
                                    </h3>
                                    <p className="font-body-medium text-[#51515E] dark:text-[#eeeef0]">
                                      Streamline your Business with our Apps.
                                      Effortlessly for every team.
                                    </p>
                                  </div>
                                  <Button
                                    style={{ marginTop: "20px" }}
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                  >
                                    See How It Works
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>

                                <div className="flex flex-wrap gap-x-14 gap-y-8 text-xs sm:text-sm">
                                  {footerColumns.map((column, index) => (
                                    <div
                                      key={index}
                                      className="flex flex-col gap-5"
                                    >
                                      <h4 className="font-label-medium text-gray-900 dark:text-white">
                                        {column.title}
                                      </h4>
                                      <ul className="flex flex-col gap-4">
                                        {column.links.map((link, linkIndex) => (
                                          <li key={linkIndex}>
                                            <div className="flex items-start gap-3">
                                              <Link
                                                href={link.url}
                                                target={
                                                  link.url.startsWith("http")
                                                    ? "_blank"
                                                    : "_self"
                                                }
                                                rel={
                                                  link.url.startsWith("http")
                                                    ? "noopener noreferrer"
                                                    : undefined
                                                }
                                                className="font-body-medium text-[#51515E] dark:text-[#d7d7de] hover:text-gray-900 dark:hover:text-white transition-colors whitespace-nowrap flex items-center gap-2"
                                              >
                                                {link.text}
                                                {column.title === "Social" && (
                                                  <FaArrowRight className="w-2 h-2" />
                                                )}
                                              </Link>

                                              {link.isNew && (
                                                <div className="hidden md:flex items-center gap-2">
                                                  <div className="w-4 h-4 flex items-center justify-center">
                                                    <div className="w-[13px] h-[13px] bg-[#c3e6ff] rounded-[6.67px] flex items-center justify-center">
                                                      <div className="w-1 h-1 bg-[#286efb] rounded-sm" />
                                                    </div>
                                                  </div>
                                                  <span className="font-body-medium text-gray-900 dark:text-white whitespace-nowrap">
                                                    New
                                                  </span>
                                                </div>
                                              )}
                                            </div>
                                          </li>
                                        ))}
                                      </ul>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-4">
                    <div className="hidden md:flex items-center gap-4">
                      <div className="w-6">
                        <ThemeSwitch />
                      </div>
                      <div className="flex items-center gap-2 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-1 shadow-sm">
                        <button
                          className="cursor-pointer px-3 py-1 text-sm font-medium text-gray-900 dark:text-white"
                          onClick={() => {
                            window.location.replace(
`https://worksuite.clikkle.com/login?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Login
                        </button>

                        <button
                          className="cursor-pointer flex items-center gap-1 rounded-xl bg-[#286efb] hover:bg-[#3367d6] px-4 py-1.5 text-sm font-medium text-white transition"
                          onClick={() => {
                            window.location.replace(
                              `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Try for free
                          <span>
                            <ArrowRightIcon className="w-4" />
                          </span>
                        </button>
                      </div>
                    </div>
                  </div>
                </nav>
              </div>
            </div>
          </section>
        </section>
        <div
          className={`py-12 max-w-[800px] mx-auto ${
            announcementOpen ? "pt-[14rem]" : "pt-12"
          }`}
        >
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <h6 className="text-sm font-semibold text-gray-500  dark:text-white">
              Legals
            </h6>
            <h1 className="text-[3rem] font-black text-gray-900 dark:text-white">
              Terms of Service
            </h1>
            <h3 className="text-[1rem] font-semibold text-gray-900 dark:text-white">
              By creating an account, you agree to our Terms of Service.
            </h3>
          </div>
        </div>

        <div className="py-12 max-w-[800px] mx-auto">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 space-y-4">
            <p className="text-gray-900 dark:text-gray-100">
              These terms and conditions (&quot;Terms&quot;,
              &quot;Agreement&quot;) are an agreement between Clikkle Inc., a
              Canadian company (&quot;Clikkle&quot;, &quot;us&quot;,
              &quot;we&quot; or &quot;our&quot;) and you (&quot;User&quot;,
              &quot;you&quot; or &quot;your&quot;). This Agreement sets forth
              the general terms and conditions of your use of the Clikkle
              website, web app, desktop app, mobile apps, browser extensions,
              and any of its products or services (collectively, &quot;App&quot;
              or &quot;Services&quot;), including Clikkle eSign, Clikkle
              Projects, Clikkle HR, Clikkle Chat, and Clikkle Campaigns.
            </p>
            <p className="text-gray-900 dark:text-gray-100">
              When you sign up for an account and agree to these Terms, the
              Agreement between you and Clikkle is formed, and the term of the
              Agreement (the “Term”) will begin. The Term will continue for as
              long as you have a Clikkle account or until you or we terminate
              the Agreement in accordance with these Terms, whichever happens
              first. If you sign up for an account on behalf of a company or
              other entity, you represent and warrant that you have the
              authority to accept these Terms and enter into the Agreement on
              its behalf.
            </p>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Accounts and Membership
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You must be at least 18 years of age to use this App. By using
                this App and by agreeing to this Agreement you warrant and
                represent that you are at least 18 years of age. If you create
                an account in the App, you are responsible for maintaining the
                security of your account and you are fully responsible for all
                activities that occur under the account and any other actions
                taken in connection with it. We may, but have no obligation to,
                monitor and review new accounts before you may sign in and use
                our Services. Providing false contact information of any kind
                may result in the termination of your account. You must
                immediately notify us of any unauthorized uses of your account
                or any other breaches of security.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                We will not be liable for any acts or omissions by you,
                including any damages of any kind incurred as a result of such
                acts or omissions. We may suspend, disable, or delete your
                account (or any part thereof) if we determine that you have
                violated any provision of this Agreement or that your conduct or
                content would tend to damage our reputation and goodwill or you
                were rude or abusive to any of our team members. If we delete
                your account for the foregoing reasons, you may not re-register
                for our Services.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Terminating your account
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You or Clikkle may terminate the Agreement at any time and for
                any reason by terminating your Clikkle account or giving notice
                to the other party. We may suspend the Service to you at any
                time, with or without cause. If we terminate your account
                without cause, and your account is a paid account, we’ll refund
                a prorated portion of any prepaid amounts for the Service.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                We won’t refund or reimburse you in any other situation,
                including if your account is suspended or terminated for cause,
                such as for a breach or violation of the Agreement.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Terminating inactive accounts
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If your account is inactive for 12 months (roughly 365 days) or
                more, we may terminate your account and you won’t be entitled to
                a refund for any prepaid amounts or reimbursement for unused
                credit. Once your account is terminated, you acknowledge and
                agree that we may permanently delete your account and all the
                data associated with it, including your workspaces.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We’re deleting accounts that are no longer active and deleting
                the user data for those accounts in order to ensure compliance
                with current and future data protection and privacy regulations.
                Deleting old data is a best practice that helps us protect user
                data and our systems. We notify affected account holders by
                sending notice(s) to the email addresses associated with the
                account’s primary contact and owner profile.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                User content
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We do not own any data, information or material
                (&quot;Content&quot;) that you submit in the App in the course
                of using the Service. You shall have sole responsibility for the
                accuracy, quality, integrity, legality, reliability,
                appropriateness, and intellectual property ownership or right to
                use of all submitted Content. We may, but have no obligation to,
                monitor and review Content in the App submitted or created using
                our Services by you. Unless specifically permitted by you, your
                use of the App does not grant us the license to use, reproduce,
                adapt, modify, publish or distribute the Content created by you
                or stored in your user account for commercial, marketing or any
                similar purpose. But you grant us permission to access, copy,
                distribute, store, transmit, reformat, display and perform the
                Content of your user account solely as required for the purpose
                of providing the Services to you. Without limiting any of those
                representations or warranties, we have the right, though not the
                obligation, to, in our own sole discretion, refuse or remove any
                Content that, in our reasonable opinion, violates any of our
                policies or is in any way harmful or objectionable.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Billing and payments
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You shall pay all fees or charges to your account in accordance
                with the fees, charges, and billing terms in effect at the time
                a fee or charge is due and payable. Where Services are offered
                on a free trial basis, payment may be required after the free
                trial period ends, and not when you enter your billing details
                (which may be required prior to the commencement of the free
                trial period). If auto-renewal is enabled for the Services you
                have subscribed for, you will be charged automatically in
                accordance with the term you selected. If, in our judgment, your
                purchase constitutes a high-risk transaction, we will require
                you to provide us with a copy of your valid government-issued
                photo identification, and possibly a copy of a recent bank
                statement for the credit or debit card used for the purchase. We
                reserve the right to change products and product pricing at any
                time.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Charges for add-ons
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If you use an add-on that has a charge, then you’ll be billed
                that additional amount with each billing cycle for as long as
                the add-on is active. The billing cycle for an add-on differ
                from the billing cycle of your base plan or other features of
                the Service, and certain add-ons may require upfront payment for
                their entire billing cycle.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Refunds
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You won’t be entitled to a refund or credit from us under any
                other circumstances. We may, at our sole discretion, offer a
                refund, discount or credit.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Billing changes
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We may change any of our pricing plans and fees at any time by
                posting a new pricing structure to our website or in your
                account and/or sending you a notification by email.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Accuracy of information
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                Occasionally there may be information in the App that contains
                typographical errors, inaccuracies or omissions that may relate
                to promotions and offers. We reserve the right to correct any
                errors, inaccuracies or omissions, and to change or update
                information or cancel orders if any information in the App or on
                any related Service is inaccurate at any time without prior
                notice (including after you have submitted your order). We
                undertake no obligation to update, amend or clarify information
                in the App including, without limitation, pricing information,
                except as required by law. No specified update or refresh date
                applied in the App should be taken to indicate that all
                information in the App or on any related Service has been
                modified or updated.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Uptime guarantee
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                We offer a Service uptime guarantee of 99% of available time per
                month. The service uptime guarantee does not apply to service
                interruptions caused by: (1) periodic scheduled maintenance or
                repairs we may undertake from time to time; (2) interruptions
                caused by you or your activities; (3) outages that do not affect
                core Service functionality; (4) causes beyond our control or
                that are not reasonably foreseeable; and (5) outages related to
                the reliability of certain programming environments.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Backups
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We perform regular backups of the Content and will do our best
                to ensure completeness and accuracy of these backups. In the
                event of the hardware failure or data loss we will restore
                backups automatically to minimize the impact and downtime.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Links to other websites
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                Although this App may link to other websites, we are not,
                directly or indirectly, implying any approval, association,
                sponsorship, endorsement, or affiliation with any linked
                website, web app or mobile apps, unless specifically stated
                herein. Some of the links in the App may be &quot;affiliate
                links&quot;. This means if you click on the link and purchase an
                item, Clikkle Inc. will receive an affiliate commission. We are
                not responsible for examining or evaluating, and we do not
                warrant the offerings of, any businesses or individuals or the
                content of their websites. We do not assume any responsibility
                or liability for the actions, products, services, and content of
                any other third-parties. You should carefully review the legal
                statements and other conditions of use of any website, web app
                or mobile apps which you access through a link from this App.
                Your linking to any other off-site websites is at your own risk.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Limitation of liability
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                To the fullest extent permitted by applicable law, in no event
                will Clikkle Inc., its affiliates, officers, directors,
                employees, agents, suppliers or licensors be liable to any
                person for (a): any indirect, incidental, special, punitive,
                cover or consequential damages (including, without limitation,
                damages for lost profits, revenue, sales, goodwill, use of
                content, impact on business, business interruption, loss of
                anticipated savings, loss of business opportunity) however
                caused, under any theory of liability, including, without
                limitation, contract, tort, warranty, breach of statutory duty,
                negligence or otherwise, even if Clikkle Inc. has been advised
                as to the possibility of such damages or could have foreseen
                such damages. To the maximum extent permitted by applicable law,
                the aggregate liability of Clikkle Inc. and its affiliates,
                officers, employees, agents, suppliers and licensors, relating
                to the services will be limited to an amount greater of one
                Canadian dollar or any amounts actually paid in cash by you to
                Clikkle Inc. for the prior one month period prior to the first
                event or occurrence giving rise to such liability. The
                limitations and exclusions also apply if this remedy does not
                fully compensate you for any losses or fails of its essential
                purpose.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Changes and amendments
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We reserve the right to modify this Agreement or its policies
                relating to the App or Services at any time, effective upon
                posting of an updated version of this Agreement in the App. When
                we do, we will revise the updated date at the bottom of this
                page. Continued use of the App after any such changes shall
                constitute your consent to such changes.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Beta services
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                From time to time, we may invite you to try products or features
                related to our Service that are not generally available to all
                of our Members or the public or we may release products or
                features related to our Service that are identified in the App
                as beta, preview, pilot, limited release, or by a similar
                designation (collectively, “Beta”). Beta may be provided for
                evaluation or testing purposes only, and, therefore, may contain
                bugs or errors, and may not be as reliable as other features of
                the Service. We may discontinue Beta services at any time in our
                sole discretion, and we may decide not to make Beta services
                generally available. For the avoidance of doubt, Beta services
                are a part of the Service, and, therefore, are subject to the
                terms of this Agreement.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Acceptance of these terms
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You acknowledge that you have read this Agreement and agree to
                all its terms and conditions. By using the App or its Services
                you agree to be bound by this Agreement. If you do not agree to
                abide by the terms of this Agreement, you are not authorized to
                use or access the App and its Services.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white mt-12">
                Contacting us
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If you would like to contact us to understand more about this
                Agreement or wish to contact us concerning any matter relating
                to it, you may send an email to{" "}
                <a
                  href="mailto:legal@clikkle.com"
                  className="text-blue-600 hover:underline"
                >
                  legal@clikkle.com
                </a>
                <h3 className="text-[1rem] font-semibold text-gray-900 dark:text-white mt-2">
                  This document was last updated on September 11, 2025.
                </h3>
              </p>
            </div>
          </div>
        </div>
      </div>

      <Footer />
      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
    </>
  );
};

export default TermOfService;
