"use client";
import React, { useEffect, useRef, useState } from "react";
import { Button } from "../components/button";
import Link from "next/link";
import Image from "next/image";
import { useTheme } from "../atoms/theme";
import { ArrowRightIcon, Ellipsis, Logs } from "lucide-react";
import { FaArrowRight } from "react-icons/fa";
import ThemeSwitch from "../components/themeSwitch";
import { apps } from "../constants/products";
import illustration from "../../public/images/banner.png";
import SidebarDrawer from "../components/sidebarDrawer";
import { footerColumns } from "../constants/footerText";
import AnnouncementBar from "../components/AnnouncementBar";
import ClikkleLogo from "@/components/ClikkleLogo";
import Footer from "@/components/Footer";
import { getCookie } from "@/utils/cookies";
import { getEncodedReferrer } from "@/utils/utils";

const PrivacyPolicy: React.FC = () => {
  const { theme } = useTheme();

  const [scrolled, setScrolled] = useState(false);
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  const [leftDrawerOpen, setLeftDrawerOpen] = useState(false);
  const toggleLeftDrawer = () => setLeftDrawerOpen((prev) => !prev);

  const [productsOpen, setProductsOpen] = useState(false);
  const [dotsOpen, setDotsOpen] = useState(false);
  const productsRef = useRef<HTMLDivElement>(null);
  const dotsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (
        productsRef.current &&
        !productsRef.current.contains(event.target as Node)
      ) {
        setProductsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (dotsRef.current && !dotsRef.current.contains(event.target as Node)) {
        setDotsOpen(false);
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);
  const [announcementOpen, setAnnouncementOpen] = useState(false);

  useEffect(() => {
    const checkHeight = () => {
      const el = document.querySelector("[data-announcement]") as HTMLElement;
      if (el && el.offsetHeight > 0) {
        setAnnouncementOpen(true);
      } else {
        setAnnouncementOpen(false);
      }
    };
    checkHeight();
    const observer = new MutationObserver(checkHeight);
    observer.observe(document.body, { childList: true, subtree: true });
    return () => observer.disconnect();
  }, []);

  useEffect(() => {
    const checkDismissedAnnouncements = () => {
      const dismissed = getCookie("dismissedAnnouncements");
      if (dismissed) {
        try {
          const dismissedIds = JSON.parse(dismissed);
          if (dismissedIds.length > 0) {
            setAnnouncementOpen(false);
          }
        } catch (error) {
          console.error("Error parsing dismissed announcements:", error);
        }
      }
    };

    checkDismissedAnnouncements();

    const interval = setInterval(checkDismissedAnnouncements, 1000);

    return () => clearInterval(interval);
  }, []);
  return (
    <>
      <div className="min-h-screen font-sans bg-white dark:bg-[#1a1a1a]">
        <section className="w-full">
          <section className="relative w-full h-28 overflow-hidden">
            <div
              className={`fixed top-0 inset-x-0 z-50 ${
                scrolled
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : productsOpen || dotsOpen
                  ? "backdrop-blur-md bg-white/90 dark:bg-[#1a1a1a]/90"
                  : ""
              }`}
            >
              <div
                ref={(el) => {
                  if (el) {
                    const height = el.getBoundingClientRect().height;

                    if (height !== 0) {
                      setAnnouncementOpen(true);
                    }
                  }
                }}
              >
                <AnnouncementBar onClose={() => setAnnouncementOpen(false)} />
              </div>
              <div className="container max-w-[1233px] mx-auto max-[1300px]:px-14 max-[1100px]:px-4">
                <nav
                  className={`relative flex items-center justify-between py-4`}
                >
                  <ClikkleLogo />

                  <div className="flex md:hidden items-center">
                    <div
                      className="flex items-center gap-0 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-[8px] shadow-sm"
                      onClick={toggleLeftDrawer}
                    >
                      <p className="text-sm font-medium text-gray-900 dark:text-white">
                        <Logs className="w-5 h-5 text-gray-900 dark:text-white mr-3" />
                      </p>

                      <p className="flex items-center  rounded-xl text-sm font-bold ">
                        Menu
                      </p>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-12 font-inter font-medium text-[18px]">
                    <Link
                      href="https://clikkle.com#features"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Features
                    </Link>
                    <Link
                      href="https://clikkle.com/#pricing"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Pricing
                    </Link>
                    <Link
                      href="/blogs"
                      className="text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                    >
                      Blog
                    </Link>
                    <div className="flex flex-row justify-center items-center">
                      <div className="relative" ref={productsRef}>
                        <button
                          onClick={() => setProductsOpen((o) => !o)}
                          className="cursor-pointer flex items-center gap-1 text-gray-900 dark:text-white hover:text-gray-900 dark:hover:text-[#d7d7de] transition-colors"
                        >
                          Products
                        </button>
                        {productsOpen && (
                          <div
                            className={`fixed ${
                              announcementOpen ? "top-[115px]" : "top-[75px]"
                            } 
              left-1/2 transform -translate-x-1/2 w-full bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
              shadow-lg p-6 border-b-2 border-gray-300 dark:border-[#575662] z-50 flex justify-center items-center pb-0`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(0, Math.ceil(apps.length / 2))
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                            </div>

                            <div className="flex flex-col gap-3">
                              {apps
                                .slice(
                                  Math.ceil(apps.length / 2),
                                  Math.ceil(apps.length / 2) + 2,
                                )
                                .map((product) => (
                                  <Link
                                    key={product.id}
                                    href={product.route}
                                    className="flex items-center gap-4 p-[0.35rem] rounded-lg hover:bg-gray-100 dark:hover:bg-white/10 transition"
                                  >
                                    <div>
                                      <Image
                                        src={product.image}
                                        alt={product.title}
                                        className="w-[26px]"
                                        width={26}
                                        height={26}
                                      />
                                    </div>
                                    <div>
                                      <p className="font-bold text-gray-900 dark:text-white text-sm">
                                        {product.title}
                                      </p>
                                      <p className="text-xs text-gray-600 dark:text-gray-300">
                                        {product.description}
                                      </p>
                                    </div>
                                  </Link>
                                ))}
                              <Link href="/products">
                                <Button
                                  style={{ marginTop: "5px" }}
                                  variant="ghost"
                                  className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                >
                                  View all products
                                  <ArrowRightIcon className="h-2 w-2 dark:text-white" />
                                </Button>
                              </Link>
                            </div>
                              <div className="hidden lg:flex flex-row items-center justify-center text-center gap-3">
                                <Image
                                  src={illustration}
                                  alt="Illustration"
                                  className="w-auto h-[175px]"
                                  width={400}
                                  height={175}
                                />
                                <div className="flex flex-col justify-center items-center h-full">
                                  <p className="text-sm text-gray-700 dark:text-gray-100 max-w-[200px] pb-5">
                                    Get started today the modern solution to
                                    power your business.
                                  </p>
                                  <Button
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                  >
                                    Get Started Today
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>

                      <div className="relative" ref={dotsRef}>
                        <Ellipsis
                          onClick={() => setDotsOpen((o) => !o)}
                          className="w-8 h-6 rounded-full p-[1px] cursor-pointer"
                        />
                        {dotsOpen && (
                          <div
                            className={`fixed ${
                              announcementOpen ? "top-[115px]" : "top-[75px]"
                            }  left-1/2 transform -translate-x-1/2 w-full 
    bg-white/90 dark:bg-[#1a1a1a]/90 backdrop-blur-md 
    shadow-lg border-b-2 border-gray-300 dark:border-[#575662] 
    z-50 flex justify-center`}
                          >
                            <div className="container max-w-[1233px] max-[1300px]:px-14 max-[1100px]:px-4 py-6">
                              <div className="flex flex-col md:flex-row justify-between items-start gap-12">
                                <div className="space-y-12">
                                  <div className="max-w-[285px] space-y-2 mb-0">
                                    <h3 className="font-instrument-sans font-medium text-[18px] leading-[20px] text-gray-900 dark:text-[#eeeef0]">
                                      Get Started For Free
                                    </h3>
                                    <p className="font-body-medium text-[#51515E] dark:text-[#eeeef0]">
                                      Streamline your Business with our Apps.
                                      Effortlessly for every team.
                                    </p>
                                  </div>
                                  <Button
                                    style={{ marginTop: "20px" }}
                                    variant="ghost"
                                    onClick={() => {
                                      window.location.replace(
                                        `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                                      );
                                    }}
                                    className={`cursor-pointer hover:bg-[#286efb] w-44 hover:dark:bg-[#286efb] hover:text-white hover:dark:text-[#eeeef0] font-inter font-[800] text-[12px] leading-[16px] tracking-[0em] rounded-2xl text-xs gap-1 px-4 py-2 h-auto bg-[#F3F4F6] dark:bg-[#1a1a1a] text-gray-900 dark:text-[#eeeef0] border border-solid border-[#8f8f8f] dark:border-[#ffffff33]`}
                                  >
                                    See How It Works
                                    <ArrowRightIcon
                                      className={`h-2 w-2 dark:text-white`}
                                    />
                                  </Button>
                                </div>

                                <div className="flex flex-wrap gap-x-14 gap-y-8 text-xs sm:text-sm">
                                  {footerColumns.map((column, index) => (
                                    <div
                                      key={index}
                                      className="flex flex-col gap-5"
                                    >
                                      <h4 className="font-label-medium text-gray-900 dark:text-white">
                                        {column.title}
                                      </h4>
                                      <ul className="flex flex-col gap-4">
                                        {column.links.map((link, linkIndex) => (
                                          <li key={linkIndex}>
                                            <div className="flex items-start gap-3">
                                              <Link
                                                href={link.url}
                                                target={
                                                  link.url.startsWith("http")
                                                    ? "_blank"
                                                    : "_self"
                                                }
                                                rel={
                                                  link.url.startsWith("http")
                                                    ? "noopener noreferrer"
                                                    : undefined
                                                }
                                                className="font-body-medium text-[#51515E] dark:text-[#d7d7de] hover:text-gray-900 dark:hover:text-white transition-colors whitespace-nowrap flex items-center gap-2"
                                              >
                                                {link.text}
                                                {column.title === "Social" && (
                                                  <FaArrowRight className="w-2 h-2" />
                                                )}
                                              </Link>

                                              {link.isNew && (
                                                <div className="hidden md:flex items-center gap-2">
                                                  <div className="w-4 h-4 flex items-center justify-center">
                                                    <div className="w-[13px] h-[13px] bg-[#c3e6ff] rounded-[6.67px] flex items-center justify-center">
                                                      <div className="w-1 h-1 bg-[#286efb] rounded-sm" />
                                                    </div>
                                                  </div>
                                                  <span className="font-body-medium text-gray-900 dark:text-white whitespace-nowrap">
                                                    New
                                                  </span>
                                                </div>
                                              )}
                                            </div>
                                          </li>
                                        ))}
                                      </ul>
                                    </div>
                                  ))}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="hidden md:flex items-center gap-4">
                    <div className="hidden md:flex items-center gap-4">
                      <div className="w-6">
                        <ThemeSwitch />
                      </div>
                      <div className="flex items-center gap-2 rounded-xl border border-gray-300 dark:border-[#ffffff3a] p-1 shadow-sm">
                        <button
                          className="cursor-pointer px-3 py-1 text-sm font-medium text-gray-900 dark:text-white"
                          onClick={() => {
                            window.location.replace(
`https://worksuite.clikkle.com/login?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Login
                        </button>

                        <button
                          className="cursor-pointer flex items-center gap-1 rounded-xl bg-[#286efb] hover:bg-[#3367d6] px-4 py-1.5 text-sm font-medium text-white transition"
                          onClick={() => {
                            window.location.replace(
                              `https://worksuite.clikkle.com/signup?theme=${theme}&referralApp=${getEncodedReferrer()}`
                            );
                          }}
                        >
                          Try for free
                          <span>
                            <ArrowRightIcon className="w-4" />
                          </span>
                        </button>
                      </div>
                    </div>
                  </div>
                </nav>
              </div>
            </div>
          </section>
        </section>
        <div className="py-12 max-w-[800px] mx-auto">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <h6 className="text-sm font-semibold text-gray-500  dark:text-white">
              Legals
            </h6>
            <h1 className="text-[3rem] font-black text-gray-900 dark:text-white">
              Privacy Policy
            </h1>
          </div>
        </div>

        <div className="py-12 max-w-[800px] mx-auto">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 space-y-8">
            <p className="text-gray-900 dark:text-gray-100">
              <strong>Effective date: </strong>September 4, 2025
            </p>
            <p className="text-gray-900 dark:text-gray-100">
              This privacy policy (&quot;Policy&quot;) describes how Clikkle
              (&quot;Clikkle&quot;, &quot;we&quot;, &quot;us&quot; or
              &quot;our&quot;) collects, protects, and uses the personal data
              that individual users and business entities (&quot;User&quot;,
              &quot;you&quot; or &quot;your&quot;) provide through the
              clikkle.com website, the related domain names, software, and any
              of its products or services, including Clikkle eSign, Clikkle
              Projects, Clikkle HR, Clikkle Chat, Clikkle Files, and Clikkle
              Campaigns (collectively, &quot;Website&quot; or
              &quot;Services&quot;). The Policy also describes the choices
              available to you regarding our use of your personal data and how
              you can access and update this information. This Policy does not
              apply to the practices of companies, websites, and services that
              we do not own or control, or to individuals that we do not employ
              or manage. Please read this Policy carefully before providing any
              personal data to us.
            </p>

            <div className="space-y-2">
              {[
                "About us",
                "Automatic collection of information",
                "Collection of personal data",
                "Managing personal data",
                "Storing personal data",
                "Information disclosure",
                "The rights of users",
                "How to exercise users’ rights",
                "Complaints",
                "Non-discrimination",
                "Billing and payments",
                "Privacy of children",
                "Newsletters and service notices",
                "Cookies and targeted advertising",
                "Do Not Track signals",
                "Links to other websites",
                "Information security",
                "Data breach",
                "Changes and amendments",
                "Contacting us",
              ].map((item, index) => (
                <p key={index} className="text-gray-900 dark:text-gray-100">
                  <strong>{index + 1}- </strong>
                  {item}
                </p>
              ))}
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                About us
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                The Website is owned and operated by Clikkle Inc., a company
                incorporated under the laws of Canada, with a registered place
                of business at 123 Bay Street, Toronto, ON M5J 2K7, Canada. We
                act in the capacity of a data controller and data processor with
                regard to the personal data processed through the Website in
                terms of the applicable data protection laws, including the
                Personal Information Protection and Electronic Documents Act
                (PIPEDA) and, where applicable, the EU General Data Protection
                Regulation (GDPR). Our role depends on the specific situation in
                which your personal data is handled by us, as explained in
                detail below:
              </p>
              <ul className="list-disc pl-6 mt-4 space-y-2 text-gray-900 dark:text-gray-100">
                <li>
                  <strong>Data controller</strong>
                  <br className="hidden sm:block" />
                  We are responsible for the collection and use of your personal
                  data through the Website and we make decisions about the types
                  of personal data that should be collected from you and
                  purposes for which such personal data should be used.
                  Therefore, we act as a data controller with regard to the
                  personal data collected directly through the Website (e.g.,
                  when you conclude a service contract with us or communicate
                  with us). We comply with the data controller’s obligations set
                  forth in the applicable laws.
                </li>
                <li>
                  <strong>Data processor</strong>
                  <br className="hidden sm:block" />
                  We act in the capacity of a data processor in situations when
                  you upload digital files (e.g., documents, images, or PDFs in
                  Clikkle eSign, Clikkle Files, or other Services) or generate
                  other data through the Website (the “Content”) and that
                  Content contains personal data. We do not own, control,
                  intentionally access, or make decisions about the Content. We
                  process the Content only in accordance with the instructions
                  issued by a respective data controller. To ensure that the
                  Content is processed in accordance with the strictest data
                  protection standards, we offer a data processing agreement
                  that is available for consultation{" "}
                  <a
                    href="https://clikkle.com/legal/dpa"
                    className="text-blue-600 hover:underline"
                  >
                    here
                  </a>
                  . To conclude the DPA, please download it, add the required
                  information, sign it, and return a copy of the signed DPA to
                  us at{" "}
                  <a
                    href="mailto:legal@clikkle.com"
                    className="text-blue-600 hover:underline"
                  >
                    legal@clikkle.com
                  </a>
                  .
                </li>
              </ul>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Automatic collection of information
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                When you visit the Website, our servers automatically record
                information that your browser sends. This data may include
                information such as your device&apos;s IP address, browser type
                and version, operating system type and version, language
                preferences, or the webpage you were visiting before you came to
                our Website, pages of our Website that you visit, the time spent
                on those pages, information you search for on our Website,
                access times and dates, and other statistics. Information
                collected automatically is used only to identify potential cases
                of abuse and establish statistical information regarding Website
                usage. This statistical information is not otherwise aggregated
                in such a way that would identify any particular user of the
                system. In most cases, such information is not considered to be
                personal data (except for your IP address). The legal basis on
                which we rely when processing your IP address is ‘pursuing our
                legitimate business interests’ (i.e., to operate, analyze, and
                protect our Website and Services, including Clikkle eSign,
                Clikkle Projects, Clikkle HR, Clikkle Chat, Clikkle Files, and
                Clikkle Campaigns). We store such data as long as it is
                necessary for analyzing and protecting our Website but no longer
                than 1 year.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Collection of personal data
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                When you use the Website or any of our Services, we collect only
                a minimal amount of personal data that is used for limited,
                specified, and legitimate purposes explicitly mentioned in this
                Policy. We do not use your personal data for other purposes that
                are different from the purposes for which it was provided.
                Below, you can find an overview of the types of personal data
                that we collect, the instances in which we do so, the purposes
                for which we use it, and the legal basis on which we rely when
                processing your personal data.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4 font-semibold">
                Personal data collected directly from you:
              </p>
              <ul className="list-disc pl-6 mt-4 space-y-4 text-gray-900 dark:text-gray-100">
                <li>
                  <strong>User account.</strong> When you create your user
                  account or request a free trial for any of our Services (e.g.,
                  Clikkle eSign, Clikkle Projects, Clikkle HR, Clikkle Chat,
                  Clikkle Files, or Clikkle Campaigns), we collect your (i)
                  first name, (ii) last name, (iii) workspace name, (iv)
                  workspace domain, (v) email address, and (vi) password. When
                  you update your user account, we collect information about
                  your work and your (i) image, (ii) date of birth, (iii)
                  biography, (iv) other email address(es), (v) phone number,
                  (vi) address(es), (vii) company name, (viii) links, (ix) tax
                  numbers, (x) logos, (xi) email information (e.g., email
                  address and sender name), and other information that you
                  decide to provide about yourself or your business. We use such
                  data to (i) register and maintain your user account, (ii)
                  enable your access to the Services, (iii) provide you with the
                  requested services, (iv) customize our services for your
                  needs, (v) contact you, if necessary, (vi) send you commercial
                  communication, and (vii) maintain our business records. The
                  legal bases on which we rely are ‘performing a contract with
                  you’ and ‘pursuing our legitimate business interests’ (i.e.,
                  analyze, grow, and administer the Website and Services). We
                  will store your personal data until your user account is
                  deleted or terminated.
                </li>
                <li>
                  <strong>Billing.</strong> When you make a payment for any of
                  our Services, you will be asked to provide (i) cardholder
                  name, (ii) credit card details (number, expiration date, CVC),
                  and (iii) billing address. We use such data to (i) process
                  your payments, (ii) issue invoices, and (iii) maintain our
                  business records. Please note that your payment data is
                  processed by our third-party payment processor Stripe and,
                  therefore, we do not store your credit card details. Stripe
                  makes available to us only a part of your payment data. The
                  legal bases on which we rely are ‘performing a contract with
                  you’ and ‘pursuing our legitimate business interests’ (i.e.,
                  to administer our business and comply with our legal
                  obligations). We will store your personal data for the time
                  period required by the applicable law (in Canada, we are
                  required to store accounting records for 6 years).
                </li>
                <li>
                  <strong>Booking a demo.</strong> When you book a demo for any
                  of our Services, we collect your (i) name, (ii) email address,
                  and (iii) any other information that you decide to provide us.
                  We use such data to (i) schedule your demo, (ii) contact you,
                  if necessary, and (iii) analyze and improve the Services. The
                  legal bases on which we rely are ‘pursuing our legitimate
                  business interests’ (i.e., to grow and promote our business)
                  and ‘your consent’ (for optional personal data). We store your
                  personal data until we deliver your demo; if you decide to
                  start using the Services, we will use your personal data to
                  register your user account.
                </li>
                <li>
                  <strong>Email enquiries.</strong> When you contact us, we
                  collect your (i) name, (ii) email address, and (iii) any
                  information that you decide to include in your message. We use
                  such data to respond to your enquiries and provide you with
                  the requested information. The legal bases on which we rely
                  are ‘pursuing our legitimate business interests’ (i.e., to
                  grow and promote our business) and ‘your consent’ (for
                  optional personal data).
                </li>
                <li>
                  <strong>Live chat.</strong> When you use the live chat
                  functionality in Clikkle Chat or other Services, we collect
                  any information that you decide to provide us. We use such
                  data to respond to your enquiries and provide you with the
                  requested information. The legal basis on which we rely is
                  ‘your consent’.
                </li>
                <li>
                  <strong>IP address.</strong> When you browse the Website, we
                  or our third-party analytics service providers collect your IP
                  address. We use your IP address to analyze the technical
                  aspects of your use of the Website and Services, prevent fraud
                  and abuse, and ensure the security of the Website. The legal
                  basis on which we rely is ‘pursuing our legitimate business
                  interests’ (i.e., to analyze and protect the Website and
                  Services). We store your IP address for no longer than 1 year.
                </li>
                <li>
                  <strong>Cookies.</strong> When you browse the Website, we
                  collect your cookie-related data. For more information about
                  our cookies and the purposes for which we use them, please
                  refer to our Cookie Policy. The legal bases on which we rely
                  are ‘pursuing our legitimate business interests’ (i.e., to
                  protect and conduct our business) and ‘your consent’ (for
                  non-essential cookies).
                </li>
              </ul>
            </div>

            <div>
              <h3 className="text-2xl font-black text-gray-900 dark:text-white my-12">
                Processing of personal data included in the Content
              </h3>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                When you upload or generate Content through our Services (e.g.,
                documents in Clikkle eSign, files in Clikkle Files, or campaigns
                in Clikkle Campaigns), we process all information that can be
                found in the Content, including any personal data the Content
                contains (for example, your clients’ names, contact details, and
                business information). We process the Content to (i) provide you
                with the requested services and (ii) perform our other
                contractual obligations. The legal basis on which we rely is
                ‘performing a contract with you’. We store such personal data
                until you stop using the Services.
              </p>
            </div>

            <div>
              <h3 className="text-2xl font-black text-gray-900 dark:text-white my-12">
                Personal data obtained from third parties
              </h3>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We may receive information about you from third parties to whom
                you have previously provided your personal data, if those third
                parties have a lawful basis for disclosing your personal data to
                us. For example, if social media login is enabled, we will
                collect the personal data that your social media provider
                discloses to us (e.g., your name, email address, and photo).
                Please note that you can control what personal data is submitted
                to us by adjusting the privacy settings of your social media
                service provider. We will use such data as described in the
                section “User account” above.
              </p>
            </div>

            <div>
              <h3 className="text-2xl font-black text-gray-900 dark:text-white my-12">
                Sensitive data
              </h3>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We do not collect or have access to any special categories of
                personal data (“sensitive data”) from you, unless you decide, at
                your own discretion, to provide such data to us. Sensitive data
                is information that relates to your health, genetics,
                biometrics, religious and political beliefs, racial origins,
                membership of a professional or trade association, sex life, or
                sexual orientation.
              </p>
            </div>

            <div>
              <h3 className="text-2xl font-black text-gray-900 dark:text-white my-12">
                Refusal to provide personal data
              </h3>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You can choose not to provide us with your personal data when
                requested, but then you may not be able to take advantage of
                some of the Website’s or Services’ features, such as Clikkle
                eSign or Clikkle Projects. Users who are uncertain about what
                information is mandatory are welcome to contact us.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Managing personal data
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You are able to delete or change certain personal data that we
                have about you. The personal data you can delete may change as
                the Website or the Services change. When you delete personal
                data, however, we may maintain a copy of the unrevised personal
                data in our records if we have a legal basis for doing so. If
                you would like to delete your personal data or permanently
                delete your account, you can do so on the settings page of your
                account on the Website.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Storing personal data
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We will retain and use your personal data for the period
                necessary to perform our contractual obligations to you, comply
                with our legal obligations, resolve disputes, and enforce our
                agreements, unless a longer retention period is required or
                permitted by law. For more details on the retention periods
                applicable to each type of personal data, please refer to
                section “Collection of personal data” above. We may use any
                aggregated data derived from or incorporating your personal data
                after you update or delete it, but not in a manner that would
                identify you personally. Once the retention period expires, your
                personal data will be securely deleted. Therefore, the right to
                access, the right to erasure, the right to rectification, and
                the right to data portability cannot be enforced after the
                expiration of the retention period.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Information disclosure
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>1) Disclosure to data processors</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                From time to time, your personal data is disclosed to our
                service providers with whom we cooperate (our data processors).
                For example, we share your personal and non-personal data with
                entities that provide certain technical support services to us,
                such as database, analytics, and email distribution services. We
                do not sell your personal data to third parties. The disclosure
                is limited to the situations when your personal data is required
                for the following purposes:
              </p>
              <ul className="list-disc pl-6 mt-2 space-y-2 text-gray-900 dark:text-gray-100">
                <li>
                  Ensuring the proper operation of the Website and Services;
                </li>
                <li>Ensuring the delivery of services that you purchase;</li>
                <li>Providing you with the requested information;</li>
                <li>Pursuing our legitimate business interests;</li>
                <li>
                  Enforcing our rights, preventing fraud, and security purposes;
                </li>
                <li>Carrying out our contractual obligations; or</li>
                <li>If you provide your prior consent to such a disclosure.</li>
              </ul>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>2) List of our data processors</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                We use a limited number of data processors. Our data processors
                agree to ensure an adequate level of protection of your personal
                data that is consistent with this privacy policy and the
                applicable data protection laws. The data processors that have
                access to your personal data are:
              </p>
              <ul className="list-disc pl-6 mt-2 space-y-2 text-gray-900 dark:text-gray-100">
                <li>
                  Our database service provider MongoDB (
                  <a
                    href="https://www.mongodb.com"
                    className="text-blue-600 hover:underline"
                  >
                    https://www.mongodb.com
                  </a>
                  ) located in the United States;
                </li>
                <li>
                  Our newsletter, marketing, and analytics service provider
                  HubSpot (
                  <a
                    href="https://www.hubspot.com"
                    className="text-blue-600 hover:underline"
                  >
                    https://www.hubspot.com
                  </a>
                  ) located in the United States;
                </li>
                <li>
                  Our payment processing service provider Stripe (
                  <a
                    href="https://stripe.com"
                    className="text-blue-600 hover:underline"
                  >
                    https://stripe.com
                  </a>
                  ) located in the United States;
                </li>
                <li>
                  Appointment scheduling service provider Calendly (
                  <a
                    href="https://calendly.com"
                    className="text-blue-600 hover:underline"
                  >
                    https://calendly.com
                  </a>
                  ) located in the United States; and
                </li>
                <li>Our independent contractors and consultants.</li>
              </ul>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>3) Disclosure of non-personal data</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                Your non-personal data may be disclosed to third parties for any
                purpose. For example, we may share it with prospects or partners
                for business or research purposes, for improving the Services,
                responding to lawful requests from public authorities, or
                developing new products and services.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>4) Legal requests</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                If we are contacted by a public authority, we may need to
                disclose information about you to the extent necessary for
                pursuing a public interest objective, such as national security
                or law enforcement.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>5) Successors</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                In case the Website or any of our Services are sold partly or
                fully, we will provide your personal data to a purchaser or
                successor entity and request the successor to handle your
                personal data in line with this Policy. We will notify you of
                any changes of the data controller.
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>6) Selling personal data</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                We do not directly sell your personal data to third parties.
                However, some of your personal data, including online
                identifiers (e.g., cookie-generated data and IP addresses) may
                be used for advertising, marketing, and monetization purposes
                (e.g., programmatic advertising, retargeting, third-party
                marketing, profiling, or cross-device tracking).
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                <strong>7) International transfers</strong>
              </p>
              <p className="text-gray-900 dark:text-gray-100 mt-2">
                Some of our data processors listed above are located outside
                Canada. For example, if you reside in Canada, we may need to
                transfer your personal data to jurisdictions outside Canada,
                such as the United States or the European Economic Area (EEA).
                In case it is necessary to make such a transfer, we will make
                sure that the jurisdiction in which the recipient third party is
                located guarantees an adequate level of protection for your
                personal data or we conclude a data processing agreement with
                the respective third party that ensures such protection. We will
                not transfer your personal data internationally if no
                appropriate level of protection can be granted.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                The rights of users
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You may exercise certain rights regarding your personal data
                processed by us. In particular, you have the right to do the
                following:
              </p>
              <ol className="list-roman pl-6 mt-4 space-y-2 text-gray-900 dark:text-gray-100">
                <li>
                  you have the right to withdraw consent where you have
                  previously given your consent to the processing of your
                  personal data;
                </li>
                <li>
                  you have the right to object to the processing of your
                  personal data if the processing is carried out on a legal
                  basis other than consent;
                </li>
                <li>
                  you have the right to learn if your personal data is being
                  processed by us, obtain disclosure regarding certain aspects
                  of the processing, and obtain a copy of the personal data
                  undergoing processing;
                </li>
                <li>
                  you have the right to verify the accuracy of your personal
                  data and ask for it to be updated or corrected;
                </li>
                <li>
                  you have the right, under certain circumstances, to restrict
                  the processing of your personal data;
                </li>
                <li>
                  you have the right, under certain circumstances, to obtain the
                  erasure of your personal data from us;
                </li>
                <li>
                  you have the right to receive your personal data processed by
                  us in a structured, commonly used, and machine-readable format
                  and, if technically feasible, to have it transmitted to
                  another controller without any hindrance;
                </li>
                <li>
                  you have the right to submit your complaint regarding our
                  processing of your personal data.
                </li>
              </ol>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                How to exercise users’ rights
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                Any requests to exercise User rights can be directed to us
                through the contact details provided at the end of this Policy.
                These requests can be exercised free of charge and will be
                addressed by us as early as possible but no later than 30 days.
                In order to verify the legitimacy of your request, we may ask
                you to provide us with an identifying piece of information that
                allows us to correctly identify you in our system. If we receive
                a request from a data subject asking to exercise the data
                subject’s rights with regard to the Content, we will forward
                such a request to the respective data controller (our client).
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Complaints
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If you would like to launch a complaint about the way in which
                we process your personal data, we kindly ask you to contact us
                first and express your concerns. If we receive your complaint,
                we will investigate it and provide you with our response as soon
                as possible. If you are not satisfied with the outcome of your
                complaint, you have the right to lodge a complaint with your
                local data protection authority, such as the Office of the
                Privacy Commissioner of Canada.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Non-discrimination
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We do not discriminate against you if you decide to exercise
                your rights. It means that we do not (i) deny any goods and
                services, (ii) charge you different prices, (iii) deny any
                discounts or benefits, (iv) impose penalties, or (v) provide you
                with lower quality Services.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Billing and payments
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We use third-party payment processor Stripe to assist us in
                processing your payment information securely for our Services.
                Stripe’s use of your personal data is governed by their privacy
                policy, which may or may not contain privacy protections as
                protective as this Policy. We suggest that you review their
                privacy policy that can be found at{" "}
                <a
                  href="https://stripe.com/en-ca/privacy"
                  className="text-blue-600 hover:underline"
                >
                  https://stripe.com/en-ca/privacy
                </a>
                .
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Privacy of children
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We do not knowingly collect any personal data from children
                under the age of 19, as per Canadian privacy laws. If you are
                under the age of 19, please do not submit any personal data
                through our Website or Services. We encourage parents and legal
                guardians to monitor their children’s Internet usage and to help
                enforce this Policy by instructing their children never to
                provide personal data through our Website or Services without
                their permission. If you have reason to believe that a child
                under the age of 19 has provided personal data to us through our
                Website or Services, please contact us.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Newsletters and service notices
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If we have your email address, we may, from time to time, send
                you a newsletter to keep you updated about the latest
                developments related to the Services, promotions, and special
                offers. You will receive our newsletters in the following
                instances:
              </p>
              <ul className="list-disc pl-6 mt-2 space-y-2 text-gray-900 dark:text-gray-100">
                <li>
                  If we receive your express (“opt-in”) consent to receive
                  marketing messages;
                </li>
                <li>If you voluntarily subscribe to our newsletter; or</li>
                <li>
                  If we decide to send you information closely related to the
                  services already purchased by you.
                </li>
              </ul>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                You can opt-out from receiving our commercial communication at
                any time free of charge by clicking on the “unsubscribe” link
                included in our newsletters or by contacting us directly (our
                contact details are available in the section “Contacting us”
                below). If necessary, we will send you important informational
                notices, such as confirmation receipts, payment information,
                technical or administrative emails, and other administrative
                updates. Please note that such notices are sent on an
                “if-needed” basis and they do not fall within the scope of
                commercial communication that may require your prior consent.
                You cannot opt-out from service-related notices.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Cookies and targeted advertising
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                The Website uses &quot;cookies&quot; to help personalize your
                online experience. For more information on our use of cookies,
                please read our Cookie Policy. In addition to using cookies and
                related technologies as described above, we also may permit
                certain third-party companies to help us tailor advertising that
                we think may be of interest to users and to collect and use
                other data about user activities on the Website. These companies
                may deliver ads that might also place cookies and otherwise
                track user behavior. You can control how such advertising is
                shown to you or opt-out from targeted advertising by managing
                your cookies as described in our Cookie Policy and consulting
                the guide powered by the Digital Advertising Alliance available
                at{" "}
                <a
                  href="https://youradchoices.com"
                  className="text-blue-600 hover:underline"
                >
                  https://youradchoices.com
                </a>
                . For more information on opting-out from advertising features
                on your device, please visit{" "}
                <a
                  href="https://www.networkadvertising.org"
                  className="text-blue-600 hover:underline"
                >
                  https://www.networkadvertising.org
                </a>
                .
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Do Not Track signals
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                Some browsers incorporate a Do Not Track feature that signals to
                websites you visit that you do not want to have your online
                activity tracked. Tracking is not the same as using or
                collecting information in connection with a website. For these
                purposes, tracking refers to collecting personal data from
                consumers who use or visit a website or online service as they
                move across different websites over time. Our Website does not
                track its visitors over time and across third-party websites.
                However, some third-party sites may keep track of your browsing
                activities when they serve you content, which enables them to
                tailor what they present to you.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Links to other websites
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                Our Website contains links to other websites that are not owned
                or controlled by us. Please be aware that we are not responsible
                for the privacy practices of such other websites or third
                parties. We encourage you to be aware when you leave our Website
                and to read the privacy statements of each and every website
                that may collect personal data.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Information security
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We secure information you provide on computer servers in a
                controlled, secure environment, protected from unauthorized
                access, use, or disclosure. We maintain reasonable
                administrative, technical, and physical safeguards in an effort
                to protect against unauthorized access, use, modification, and
                disclosure of personal data in our control and custody. We
                implement the following security measures: secured networks; use
                of virtual private networks (VPN); encryption; SSL protocol;
                secured calls; strong passwords; limited access to your personal
                data by our staff; anonymization of personal data (when
                possible); and carefully selected data processors. However, no
                data transmission over the Internet or wireless network can be
                guaranteed. Therefore, while we strive to protect your personal
                data, you acknowledge that (i) there are security and privacy
                limitations of the Internet which are beyond our control; (ii)
                the security, integrity, and privacy of any and all information
                and data exchanged between you and our Website cannot be
                guaranteed; and (iii) any such information and data may be
                viewed or tampered with in transit by a third-party, despite
                best efforts.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Data breach
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                In the event we become aware that the security of the Website or
                any of our Services has been compromised or users’ personal data
                has been disclosed to unrelated third parties as a result of
                external activity, including, but not limited to, security
                attacks or fraud, we reserve the right to take reasonably
                appropriate measures, including, but not limited to,
                investigation and reporting, as well as notification to and
                cooperation with law enforcement authorities. In the event of a
                data breach, we will make reasonable efforts to notify affected
                individuals if we believe that there is a reasonable risk of
                harm to the user as a result of the breach or if notice is
                otherwise required by law. When we do, we will post a notice on
                the Website and send you an email.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Changes and amendments
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                We may update this Policy from time to time in our discretion
                and will notify you of any material changes to the way in which
                we treat personal data. When changes are made, we will revise
                the updated date at the top of this page. We may also provide
                notice to you in other ways at our discretion, such as through
                contact information you have provided. Any updated version of
                this Policy will be effective immediately upon the posting of
                the revised Policy unless otherwise specified. Your continued
                use of the Website or Services after the effective date of the
                revised Policy (or such other act specified at that time) will
                constitute your consent to those changes. However, we will not,
                without your consent, use your personal data in a manner
                materially different than what was stated at the time your
                personal data was collected.
              </p>
            </div>

            <div>
              <h2 className="text-4xl font-black text-gray-900 dark:text-white my-12">
                Contacting us
              </h2>
              <p className="text-gray-900 dark:text-gray-100 mt-4">
                If you would like to contact us to understand more about this
                Policy or wish to contact us concerning any matter relating to
                individual rights and your personal data, you may use the
                following contact details:
                <br className="hidden sm:block" />
                Email address:{" "}
                <a
                  href="mailto:legal@clikkle.com"
                  className="text-blue-600 hover:underline"
                >
                  legal@clikkle.com
                </a>
                <br className="hidden sm:block" />
                Postal address: 6-470 king st west suite 255 Oshawa ON L1J 2K9
                Canada
              </p>
            </div>
          </div>
        </div>
      </div>

      <Footer />
      <SidebarDrawer open={leftDrawerOpen} toggle={toggleLeftDrawer} />
    </>
  );
};

export default PrivacyPolicy;
